/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set sw=2 ts=8 et ft=cpp : */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_MediaTaskUtils_h
#define mozilla_MediaTaskUtils_h

#include "nsThreadUtils.h"

// The main reason this file is separate from MediaUtils.h
#include "base/task.h"

namespace mozilla {
namespace media {

/* media::NewTaskFrom() - Create a Task from a lambda.
 *
 * Similar to media::NewRunnableFrom() - Create an nsRunnable from a lambda,
 * but ignore the return value from the lambda.
 *
 * Prefer NS_NewRunnableFunction(), which provides a specific name, unless the
 * lambda really must have a non-void return value that is to be ignored.
 */

template <typename OnRunType>
class LambdaTask : public Runnable {
 public:
  explicit LambdaTask(OnRunType&& aOnRun)
      : Runnable("media::LambdaTask"), mOnRun(std::move(aOnRun)) {}

 private:
  NS_IMETHOD
  Run() override {
    mOnRun();
    return NS_OK;
  }
  OnRunType mOnRun;
};

template <typename OnRunType>
already_AddRefed<LambdaTask<OnRunType>> NewTaskFrom(OnRunType&& aOnRun) {
  typedef LambdaTask<OnRunType> LambdaType;
  RefPtr<LambdaType> lambda = new LambdaType(std::forward<OnRunType>(aOnRun));
  return lambda.forget();
}

}  // namespace media
}  // namespace mozilla

#endif  // mozilla_MediaTaskUtils_h
