// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_PUBLIC_CPP_EVENT_REWRITER_CONTROLLER_H_
#define ASH_PUBLIC_CPP_EVENT_REWRITER_CONTROLLER_H_

#include <memory>

#include "ash/ash_export.h"
#include "ui/events/ash/event_rewriter_ash.h"

namespace ui {
class Event;
class EventRewriter;
}  // namespace ui

namespace ash {

class AccessibilityEventRewriterDelegate;

// Allows clients to toggle some event rewriting behavior.
class ASH_EXPORT EventRewriterController {
 public:
  // Returns the singleton EventRewriterController instance.
  static EventRewriterController* Get();

  // Initializes this controller after ash::Shell finishes initialization.
  virtual void Initialize(
      ui::EventRewriterAsh::Delegate* event_rewriter_delegate,
      AccessibilityEventRewriterDelegate*
          accessibility_event_rewriter_delegate) = 0;

  // Takes ownership of |rewriter| and adds it to the current event sources.
  virtual void AddEventRewriter(
      std::unique_ptr<ui::EventRewriter> rewriter) = 0;

  // Enables the KeyboardDrivenEventRewriter, which is disabled by default.
  // This only applies when the user is on the login screen.
  virtual void SetKeyboardDrivenEventRewriterEnabled(bool enabled) = 0;

  // If true, Shift + Arrow keys are rewritten to Tab/Shift-Tab keys.
  // This only applies when the KeyboardDrivenEventRewriter is active.
  virtual void SetArrowToTabRewritingEnabled(bool enabled) = 0;

  // Continue dispatch of key events that were unhandled by ChromeVox.
  virtual void OnUnhandledSpokenFeedbackEvent(
      std::unique_ptr<ui::Event> event) = 0;

  // Discards key events and sends to spoken feedback when true.
  virtual void CaptureAllKeysForSpokenFeedback(bool capture) = 0;

  // Sends mouse events to accessibility component extensions when true.
  virtual void SetSendMouseEvents(bool value) = 0;

  // Either propagates or cancels a stored key event for ChromeVox in mv3.
  virtual void ProcessPendingSpokenFeedbackEvent(unsigned int id,
                                                 bool propagate) = 0;

  // Enables or disables key event handling for the ChromeVox in mv3.
  virtual void SetSpokenFeedbackMv3KeyHandlingEnabled(bool enabled) = 0;

 protected:
  virtual ~EventRewriterController() {}
};

}  // namespace ash

#endif  // ASH_PUBLIC_CPP_EVENT_REWRITER_CONTROLLER_H_
